/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/


package cnrg.itx.datax.devices;

/**
 * Class to represent an RTP packet. This will contain the RTP header and the
 * audio data byte array.
 */
class RTPPacket
{
	/**
	 * Attribute to store the RTP header for the packet.
	 */
	private RTPHeader header;
	
	/**
	 * Attribute to store the audio data as a byte array.
	 */
	private byte[] data;
	
	/**
	 * Default constructor for an RTP packet.
	 */
	public RTPPacket()
	{
		header = null;
		data = null;
	}
	
	/**
	 * Constructor for making an RTP packet from its constituents.
	 * @param header the RTP header for this packet
	 * @param data the byte array containing the audio data
	 */
	public RTPPacket(RTPHeader header, byte[] data)
	{
		this.header = header;
		this.data = data;
	}
	
	/**
	 * Constructor for making a RTP packet out of a byte array that
	 * represents a RTP packet.
	 * @param rtpData the array of bytes representing a RTP packet
	 */
	public RTPPacket(byte[] rtpData)
	{
		// Initialize the header with the RTP data packet
		header = new RTPHeader(rtpData);
		
		int headerLen = header.getHeaderSize();
		data = new byte[rtpData.length-headerLen];
		
		// Copy the data part of the packet into the data array
		System.arraycopy(rtpData, headerLen, data, 0, data.length);
	}
	
	/**
	 * Accessor for the RTP header.
	 * @return RTPHeader the RTP header for this packet
	 */
	public RTPHeader getRTPHeader()
	{
		return header;
	}
	
	/**
	 * Mutator for the RTP header.
	 * @param RTPHeader the new RTP header for this packet
	 */
	public void setRTPHeader(RTPHeader header)
	{
		this.header = header;
	}
	
	/**
	 * Accessor for the audio data byte array.
	 * @return byte[] the array of audio data bytes
	 */
	public byte[] getData()
	{
		return data;
	}
	
	/**
	 * Mutator for the audio data byte array.
	 * @param byte[] the array of audio data bytes
	 */
	public void setData(byte[] data)
	{
		this.data = data;
	}
	
	/**
	 * Method to get the byte array form of an RTP packet.
	 * @return byte[] the byte array of the RTP packet
	 */
	public byte[] toByteArray()
	{
		int headerSize = header.getHeaderSize();
		
		// Allocate the byte array of size header + data
		byte[] packet = new byte[headerSize + data.length];

		// Copy the header into the byte array
		header.copyHeader(packet);
		
		// Copy the data into the byte array
		System.arraycopy(data, 0, packet, headerSize, data.length);

		return packet;
	}
}
